		function [G, Z, P] = IMPULSE_2_P_Z(h, Norder)
		
		%	Computes the poles and zeros for a recursive digital filter from at least 
		%	2*Norder+1 values of the impulse response where Norder is the filter order.
		%	
		%	h	The vector contains the impulse response and must contain
		%		at least 2*Norder+1 values. The leading value may not be zero.		
		
		% 		The algorithm is described in ERIKSSON/WANHAMMAR: TIDSDISKRETA FILTER,
		%		VOL. 2, LINKOPING UNIV. 1978. (In Swedish).
 		%		
		%		See also imp2tf.m
		
		% 	Toolbox for DIGITAL FILTERS USING MATLAB
		
		% Author: 			Lars Wanhammar  1978-10-17
		% Modified by: 		LW, 2004-11-02, 2005-11-11
		% 	Copyright:		by authors - not released for commercial use
		% Version: 			1
		% Known bugs:		May yield inaccurate results for about N > 12 and higly clustered poles.
		% Report bugs to:	Wanhammar@gmail.com
		
		while h(1) == 0 
			h(1) = [];
		end
		G = h(1);
		h0 = h(:); 	
		m = fix(length(h0)/2); 
		Hankel = hankel(h0(2:m+1), h0(m+1:2*m));
		n = rank(Hankel);	% Estimate the order; often too high
		%  n = Norder;
		% Compute the denominator
		Hd = Hankel(1:n, 1:n); 
		h = h0(n+2:2*n+1);
		denum = [1 fliplr((-Hd\h)')]';
		
		% Compute the numerator
		Hn = toeplitz(h0(1:n+1), [h0(1) zeros(1, n)]);
		numer = Hn*denum;
		
		if sum(isnan(numer)) ~= 0
			return
		else
			Z = cplxpair(roots(numer));
			P = cplxpair(roots(denum));	
			% Remove cancelling poles and zeros  
			[Z, P] = PZ_CANCEL(Z, P); 
		end		
	
	
	function [Zc, Pc] = PZ_CANCEL(Z, P, Tol)
	
	%	Removes any canceling poles and zeros that have a 
	%	difference in magnitude that is less than Tol.
  
	% 	Toolbox for DIGITAL FILTERS USING MATLAB
	
	% 	Author: 		Oscar Gustafsson, 2009-07-15
	% 	Modified by: 	LW 2009-07-21
	% 	Copyright:		by authors - not released for commercial use
	% 	Version: 		1
	% 	Known bugs:	 
	% 	Report bugs to:	Wanhammar@gmail.com
 
	if nargin < 3
		Tol = 1e-6;
	end
	pc = [];
	zc = [];
	for k = 1:length(P)
		for l = 1:length(Z)
			if(abs(P(k)-Z(l)) <= Tol)
				pc = [pc k];
				zc = [zc l];
			end
		end
	end
	Pc = P;
	Pc(pc) = [];
	Zc = Z;
	Zc(zc) = [];
